/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Qt includes
#include <QString>

/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{

/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: CalcOptions
 */
class DECLSPEC CalcOptions: public QObject
{
  Q_OBJECT

  Q_PROPERTY(
    bool isDeepCalculation WRITE setDeepCalculation READ isDeepCalculation)
  Q_PROPERTY(Enums::MassType massType WRITE setMassType READ getMassType)
  Q_PROPERTY(Enums::CapType capType WRITE setCapType READ getCapType)
  Q_PROPERTY(Enums::ChemicalEntity monomerEntities WRITE setMonomerEntities READ
               getMonomerEntities)
  Q_PROPERTY(Enums::ChemicalEntity polymerEntities WRITE setPolymerEntities READ
               getPolymerEntities)
  Q_PROPERTY(Enums::CapType selectionType WRITE setCapType READ getCapType)

  public:
  Q_INVOKABLE explicit CalcOptions(QObject *parent = nullptr);
  Q_INVOKABLE explicit CalcOptions(bool deep_calculation,
                                   Enums::MassType mass_type,
                                   Enums::CapType selectionping,
                                   Enums::ChemicalEntity monomer_entities,
                                   Enums::ChemicalEntity polymer_entities,
                                   QObject *parent = nullptr);

  // Pseudo copy constructor
  Q_INVOKABLE explicit CalcOptions(const CalcOptions &other,
                                   QObject *parent = nullptr);
  Q_INVOKABLE CalcOptions &initialize(const CalcOptions &other);
  Q_INVOKABLE CalcOptions *clone(QObject *parent = nullptr);
  Q_INVOKABLE static CalcOptions *clone(const CalcOptions &other,
                                        QObject *parent = nullptr);

  virtual ~CalcOptions();

  Q_INVOKABLE void
  setIndexRange(const IndexRange &sequence_range = IndexRange());
  Q_INVOKABLE void setIndexRange(qsizetype index_start, qsizetype index_end);
  Q_INVOKABLE void setIndexRanges(const IndexRangeCollection &sequence_ranges);
  Q_INVOKABLE const IndexRangeCollection &getIndexRangeCollectionCstRef() const;
  Q_INVOKABLE IndexRangeCollection &getIndexRangeCollectionRef();
  Q_INVOKABLE IndexRangeCollection *getIndexRangeCollection();

  Q_INVOKABLE void setDeepCalculation(bool deep);
  Q_INVOKABLE bool isDeepCalculation() const;

  Q_INVOKABLE void setMassType(Enums::MassType mass_type);
  Q_INVOKABLE Enums::MassType getMassType() const;

  Q_INVOKABLE void setSelectionType(Enums::SelectionType selection_type);
  Q_INVOKABLE Enums::SelectionType getSelectionType() const;

  Q_INVOKABLE void setCapType(Enums::CapType selection_type);
  Q_INVOKABLE Enums::CapType getCapType() const;

  Q_INVOKABLE void setMonomerEntities(Enums::ChemicalEntity monomer_chem_ent);
  Q_INVOKABLE Enums::ChemicalEntity getMonomerEntities() const;

  Q_INVOKABLE void setPolymerEntities(Enums::ChemicalEntity polymer_chem_ent);
  Q_INVOKABLE Enums::ChemicalEntity getPolymerEntities() const;

  Q_INVOKABLE CalcOptions &operator=(const CalcOptions &other) = delete;
  Q_INVOKABLE bool operator==(const CalcOptions &other) const;
  Q_INVOKABLE bool operator!=(const CalcOptions &other) const;

  Q_INVOKABLE QString toString() const;

  static void registerJsConstructor(QJSEngine *engine);

  protected:
  // Tells if the calculation should involve the recalculation of
  // all the masses of the monomers of the sequence.
  bool m_deepCalculation = false;

  Enums::MassType m_massType = Enums::MassType::BOTH;

  Enums::CapType m_capType = Enums::CapType::BOTH;

  Enums::ChemicalEntity m_monomerEntities = Enums::ChemicalEntity::NONE;

  Enums::ChemicalEntity m_polymerEntities = Enums::ChemicalEntity::NONE;

  Enums::SelectionType m_selectionType = Enums::SelectionType::OLIGOMERS;

  IndexRangeCollection *mp_indexRangeCollection = nullptr;
};

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: CalcOptions
 */

} // namespace libXpertMassCore
MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, CalcOptions)
} // namespace MsXpS


Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::CalcOptions);
extern int calcOptionsMetaTypeId;
